package adf.sample.poicc;

import javax.servlet.ServletContext;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import oracle.adf.share.ADFContext;
import oracle.adf.share.config.SiteCC;
import oracle.adf.share.security.SecurityContext;

import oracle.mds.core.MetadataObject;
import oracle.mds.core.RestrictedSession;
import oracle.mds.core.SessionOptions;

import oracle.security.jps.ResourcePermission;


/**
 * Custom implementation of SiteCC that reads its site information from a context
 * parameter in the web.xml file.
 *
 * <context-param>
 * <param-name>adf.sample.cc.Site</param-name>
 * <param-value>true</param-value>
 * </context-param>
 *
 * If the user has a specific JAAS permission granted then he/she could use the
 * request URL to override the context parameter value, which is useful for testing
 * The URL request parameter name needs to be "siteCC".
 *
 * This solution uses a servlet filter that is configured for the ADF view layer
 * project. The servlet filter must be configued after the ADF binding filter to
 * ensure that the ADF binding context is created and the MDS context is accessible.
 *
 * <!-- web.xml context parameter read by the customization class to determine the
 * site value. Allowed values:  america, emea, apac , reset-->
 * <context-param>
 *   <param-name>adf.sample.cc.Site</param-name>
 *   <param-value>americas</param-value>
 * </context-param>
 *
 * <!-- filter to read from the page request. If the sitecc parameter is part of
 * the request URL then this value overrides the value read from the web.xml
 * context parameter if the user has permission to do so -->
 *
 * <filter>
 * <filter-name>SiteCCFilter</filter-name>
 * <filter-class>adf.sample.poicc.servlet.DynamicSiteCCFilter</filter-class>
 * </filter>
 * ...
 *
 * <filter-mapping>
 * <filter-name>SiteCCFilter</filter-name>
 * <url-pattern>/faces/*</url-pattern>
 * </filter-mapping>
 *
 * The user who overrides the context parameter with a request parameter must have
 * the oracle.security.jps.ResourcePermission granted that is shown below
 *
 * <permission>
 * <class>oracle.security.jps.ResourcePermission</class>
 * <name>resourceType=Customization,resourceName=adf.sample.cc.Site</name>
 * <actions>override</actions>
 * </permission>
 *
 * @author Frank Nimphius
 */
public class DynamicSiteCC extends SiteCC{
    final String ctxParameter                  = "adf.sample.cc.Site";
    
    //the permission needs to be created in the jazn-data.xml file of the 
    //developed application. The permission class is 
    
    public static final String ACTION = "override";
    public static final String RESOURCE_NAME = "adf.sample.cc.Site";
    public static final String RESOURCE_TYPE = "Customization";
    final String SITE_CC_SESSION_ATTIBUTE = "adf.sample.cc.SiteCC.value";
    
    public DynamicSiteCC() {
        super();
    }

    @Override
    public String[] getValue(RestrictedSession restrictedSession,
                             MetadataObject metadataObject) {
        //accessing the MDS session
        SessionOptions so = restrictedSession.getSessionOptions();
        ServletContext servletCtx = null;
        servletCtx = (ServletContext) so.getServletContextAsObject();
        
        //read site information from context parameter
        Object siteCCValue = null;
        siteCCValue = servletCtx.getInitParameter(ctxParameter);
        
        //try to read session attribute to check if site information is 
        //overridden using the siteCC URL request parameter
        
        ADFContext  adfCtx = ADFContext.getCurrent();
        Object      _request = adfCtx.getEnvironment().getRequest();
        Object siteCCSessionAttribute  = null;
        
        if (_request != null && _request instanceof HttpServletRequest) {                  
            HttpSession _session = null;
            _session = ((HttpServletRequest)_request).getSession(true);
            siteCCSessionAttribute  =  _session.getAttribute(SITE_CC_SESSION_ATTIBUTE);
        }
        
        if(siteCCSessionAttribute !=null && isAllowedOverride()){
           //site information is overridded
           siteCCValue = siteCCSessionAttribute;
        }
        
        //if siteValue is set, return it
        if (siteCCValue != null){
           return new String[]{(String)siteCCValue};
        }        
        return new String[0];
    }    
    
    /**
     * Private method using the ADF Security Context to check the OPS Resource 
     * Permission for The Site administration permission. The ResourcePermission
     * class is from OPSS and can be used as a generic permission
     * @return true/false if user is authorized to override the site level customization
     */
    private boolean isAllowedOverride(){        
      boolean hasPermissionGranted = false;
      ADFContext adfCtx = ADFContext.getCurrent();
      SecurityContext securityCtx = adfCtx.getSecurityContext();        
      ResourcePermission sitePermission = null;
      sitePermission = new ResourcePermission(RESOURCE_TYPE,RESOURCE_NAME,ACTION);        
      hasPermissionGranted = securityCtx.hasPermission(sitePermission);    
      return hasPermissionGranted;
    }
}
