package adf.sample.view.beans;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.el.ELContext;
import javax.el.ExpressionFactory;
import javax.el.MethodExpression;

import javax.faces.component.UIComponent;
import javax.faces.component.UIViewRoot;
import javax.faces.context.FacesContext;

import oracle.adf.model.BindingContext;
import oracle.adf.model.binding.DCIteratorBinding;
import oracle.adf.view.rich.component.rich.data.RichTree;
import oracle.adf.view.rich.context.AdfFacesContext;

import oracle.binding.BindingContainer;
import oracle.binding.OperationBinding;

import oracle.jbo.Row;
import oracle.jbo.uicli.binding.JUCtrlHierBinding;
import oracle.jbo.uicli.binding.JUCtrlHierNodeBinding;
import oracle.jbo.uicli.binding.JUCtrlHierTypeBinding;

import org.apache.myfaces.trinidad.event.SelectionEvent;
import org.apache.myfaces.trinidad.model.CollectionModel;
import org.apache.myfaces.trinidad.model.RowKeySet;
import org.apache.myfaces.trinidad.model.RowKeySetImpl;


/**
 *  TreeFormHelperBean is a managed bean configured in view scope to survive
 *  a request. It is in viewScope because it keeps track of the af:switcher 
 *  state. Because no component binding references this class, it is okay
 *  to have it in viewScope. To avoid component bindings, the tree component
 *  is either looked up from the selection event object or the UIViewRoot. 
 *  Same for the panelGroup that surrounds the af:switcher component
 */
public class TreeFormHelperBean {

    //keeps track of current selected facet
    String switcherFacet = "";

    public TreeFormHelperBean() {
        super();
    }

    public void setSwitcherFacet(String switcherFacet) {
        this.switcherFacet = switcherFacet;
    }

    public String getSwitcherFacet() {
        return switcherFacet;
    }

    /*
     * Method that decorates the default ADF tree selection listener. It 
     * is responsible for setting the state of the switcher property so 
     * correct input form (Department or Employee) is shown
     */
    public void treeSelectionHandler(SelectionEvent selectionEvent) {

        //preserve default selection functionality by calling El below
        //#{bindings.allDepartmentsWithEmployees.treeModel.makeCurrent}
        //Do so using generic code that does not fail if binding names
        //in the PageDef file change

        RichTree tree = (RichTree)selectionEvent.getSource();
        CollectionModel model = (CollectionModel)tree.getValue();
        JUCtrlHierBinding treeBinding =
            (JUCtrlHierBinding)model.getWrappedData();
        
        //the EL code below is generic because the tree binding name is read
        //from the treebinding itself. Using this EL approach ensures that the
        //DataSource feature of the tree - which is that the tree nodes synch
        //up with an independent iterator in the page - continues working
        String methodExpression =
            "#{bindings." + treeBinding.getName() + ".treeModel.makeCurrent}";
        //resolve method expression in private function
        this.handleTreeSelection(selectionEvent, methodExpression);
        //access selected rowKey. Because the tree is configured for single node
        //selection, we take the first entry in the RowKeySet if there is one
        RowKeySet rks = selectionEvent.getAddedSet();
        Iterator iterator = rks.iterator();
        if (iterator.hasNext()) {
            List rowKey = (List)iterator.next();
            //get access to the ADF tree binding
            JUCtrlHierNodeBinding node = treeBinding.findNodeByKeyPath(rowKey);

            //determine the node the user selected. We determine this by the
            //Object that represnets the node, which is the full package and
            //class name of the View Object
            JUCtrlHierTypeBinding nodeType = node.getHierTypeBinding();
            String defName = nodeType.getStructureDefName();
            switcherFacet = defName;
            //refresh the pabek group containing the switcher component. The
            //argument passed to the method is the id of the panel group. If the
            //group is contained in a nmaing container, then the container ID needs
            //to be prefixed separated by a colon (:)
            partiallyRefreshUIComponent("pgl1");

        }
    }

    /**
     * Method is invoked by the "Create Sibling" button on the Department form. It
     * calls the Create operation binding configured in the PageDef file for the 
     * View Object that is used to build the tree root and the Department form
     * @return null
     */
    public String onCreateDepartment() {
        BindingContainer bindings = getBindings();
        OperationBinding operationBinding =
            bindings.getOperationBinding("Create");
        Object result = operationBinding.execute();
        if (!operationBinding.getErrors().isEmpty()) {
            //TODO add error handling
            return null;
        }

        return null;
    }

    /**
     * Method called from the Create Sibling button on the employee form. It 
     * calls the Create1 binding defined in the PageDef file. The Create1 binding
     * is created by dragging the Create operation binding from the View Object
     * that is used to build the employee form. Note that the View Object is not 
     * dependent on the allDepartments View Object
     * @return null
     */
    public String onCreateEmployee() {
        BindingContainer bindings = getBindings();
        OperationBinding operationBinding =
            bindings.getOperationBinding("Create1");
        Object result = operationBinding.execute();
        if (!operationBinding.getErrors().isEmpty()) {
            //TODO add error handling
            return null;
        }

        //set department Id to the value of the tree parent node
        DCIteratorBinding departmentIter =
            (DCIteratorBinding)bindings.get("allDepartmentsWithEmployeesIterator");
        Row currentDepartmentRow = departmentIter.getCurrentRow();
        Object deptId = currentDepartmentRow.getAttribute("DepartmentId");
        DCIteratorBinding employeesIter =
            (DCIteratorBinding)bindings.get("allEmployeesIterator");
        Row currentEmployeeRow = employeesIter.getCurrentRow();
        currentEmployeeRow.setAttribute("DepartmentId", deptId);

        return null;
    }

    /**
     * Method is invoked from the context menu to create a new department
     * @return null
     */
    public String onCreateDepartmentFromMenu() {
        switcherFacet = "adf.sample.model.vo.DepartmentsView";
        onCreateDepartment();
        partiallyRefreshUIComponent("pgl1");
        return null;
    }

    /**
     * Method is called from the context menu to create a new employee for the
     * selected department
     * @return null
     */
    public String onCreateEmployeeForSelectedDepartmentFromMenu() {
        switcherFacet = "adf.sample.model.vo.EmployeesView";
        onCreateEmployee();
        partiallyRefreshUIComponent("pgl1");
        return null;
    }

    /**
     * whenever the department form is submitted, this method ensures the 
     * edited record is shown as selected in the tree and the tree is 
     * refreshed
     * @return null
     */
    public String onSubmitDepartmentForm() {
        RowKeySet selectedRowKeys = new RowKeySetImpl();
        BindingContainer bindings = getBindings();

        DCIteratorBinding departmentIter =
            (DCIteratorBinding)bindings.get("allDepartmentsWithEmployeesIterator");
        Row currentDepartmentRow = departmentIter.getCurrentRow();

        ArrayList list = new ArrayList();
        list.add(currentDepartmentRow.getKey());
        selectedRowKeys.add(list);

        RichTree tree = findTreeComponentOnPage("t1");
        if (tree != null) {
            tree.setSelectedRowKeys(selectedRowKeys);            
            partiallyRefreshUIComponent(tree);
        }
        return null;
    }

    /**
     * whenever the employee form is submitted, this method ensures the 
     * edited record is shown as selected in the tree and the tree is 
     * refreshed
     * @return
     */
    public String onSubmitEmployeeForm() {

        RowKeySet selectedRowKeys = new RowKeySetImpl();
        RowKeySet disclosedRowKeys = new RowKeySetImpl();
        BindingContainer bindings = getBindings();

        DCIteratorBinding employeesIter =
            (DCIteratorBinding)bindings.get("allEmployeesIterator");
        Row currentEmployeeRow = employeesIter.getCurrentRow();

        DCIteratorBinding departmentIter =
            (DCIteratorBinding)bindings.get("allDepartmentsWithEmployeesIterator");
        Row currentDepartmentRow = departmentIter.getCurrentRow();
        
        //determine selected row key and disclosed row key (parent node). In a 
        //two level tree as in this sample, its easy to do this by getting this
        //information from the iterators. However in a complex tree you want to 
        //find a generic example similar to what Lynn Munsinger and I describe 
        //in chapter 9 of the Fusion Developer Guide book published by McGraw Hill 
        //in 2010
        ArrayList list = new ArrayList();
        list.add(currentDepartmentRow.getKey());
        list.add(currentEmployeeRow.getKey());
        selectedRowKeys.add(list);        
        
        ArrayList disclosedList = new ArrayList();
        disclosedList.add(currentDepartmentRow.getKey());      
        disclosedRowKeys.add(disclosedList);
  
        //get the tree component by ID. Make sure that the ID of surrounding
        //naming containers is included as a prefix

        RichTree tree = findTreeComponentOnPage("t1");
        if (tree != null) {
            tree.setSelectedRowKeys(selectedRowKeys);           
           //ensure the department node gets disclosed, which we do by
           //adding the node to the list of disclosed rowKeys          
           tree.setDisclosedRowKeys(disclosedRowKeys);
            partiallyRefreshUIComponent(tree);
        }

        return null;
    }

    /* ******************************************************************
   * PRIVATE METHODS
   * *****************************************************************/

    private BindingContainer getBindings() {
        return BindingContext.getCurrent().getCurrentBindingsEntry();
    }

    //helper method to synch the current tree node selection with the
    //ADF binding
    private void handleTreeSelection(SelectionEvent selectionEvent,
                                     String el) {
        FacesContext fctx = FacesContext.getCurrentInstance();
        ELContext elctx = fctx.getELContext();
        ExpressionFactory exprFactory =
            fctx.getApplication().getExpressionFactory();

        MethodExpression me =
            exprFactory.createMethodExpression(elctx, el, Object.class,
                                               new Class[] { SelectionEvent.class });
        me.invoke(elctx, new Object[] { selectionEvent });

    }

    //quick and dirty approach to find a component in the view root. There are 
    //better options for this, but this here does for what we need in this sample
    private RichTree findTreeComponentOnPage(String treeId) {
        FacesContext fctx = FacesContext.getCurrentInstance();
        UIViewRoot viewRoot = fctx.getViewRoot();
        UIComponent component = viewRoot.findComponent(treeId);
        if (component != null && component instanceof RichTree) {
            return (RichTree)component;
        }

        return null;
    }
   
    //refreshes component by passed on component id. Be aware of naming containers!
    //Components in a naming container must have a prefix "<namingContainerId>:"
    private void partiallyRefreshUIComponent(String uid) {
        FacesContext fctx = FacesContext.getCurrentInstance();
        UIViewRoot viewRoot = fctx.getViewRoot();
        UIComponent component = viewRoot.findComponent(uid);
        partiallyRefreshUIComponent(component);
    }

    private void partiallyRefreshUIComponent(UIComponent component) {
        AdfFacesContext adfFacesContext = AdfFacesContext.getCurrentInstance();
        if (component != null) {
            adfFacesContext.addPartialTarget(component);
        }
    }

}
