/*
 * Created on Dec 13, 2004  
 */
package com.saternos.database.utilities;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.util.ArrayList;
import java.util.List;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;

/**
 * @author Casimir Saternos
 * @version 1.0
 * 
 *         This program is run at the command line. Given a excel
 *         spreadsheet, a script is generated to create the external table(s)
 *         that references the data from the spreadsheet.
 */
public class ExternalTableGenerator {

	static String newline="\r\n";
	
	public static void main(String args[]) {
		
		if (args.length != 1) {
			System.out.println(newline+"usage: Create_external_table <excel_file_name>"+newline);
			System.exit(0);
		}
		System.out.println("Begin processing.");
		
		ExternalTableGenerator generator = new ExternalTableGenerator(args[0]);

		System.out.println("Using working directory " + new File(generator.pwd).getAbsolutePath());
		
		generator.execute();
		
		System.out.println("Processing complete.");

	}
	
	// All sheets in the workbook use the following constants.
	
	/**
	 * The index of the row where the values that are used
	 * for the names of the table columns are retrieved
	 */
	private final int COLUMN_NAME_ROW = 0;

	/**
	 * The index of the row where the values that are used
	 * for the types of the table columns are retrieved
	 * (VARCHAR2 and NUMBER are the only two types currently
	 * handled).
	 */
	private final int COLUMN_TYPE_ROW = 1;

	/**
	 * List of external table definitions
	 */
	private List externalTables;

	/**
	 * Present working directory
	 */
	private String pwd;

	/**
	 * The Excel Spreadsheet (xls) that is being accessed
	 */
	private String spreadsheet;

	private String ddlString ="";
	
	/**
	 * @param string
	 */
	public ExternalTableGenerator(String spreadsheet) {

		pwd = new File("").getAbsolutePath();
		
		this.spreadsheet = pwd+File.separator+spreadsheet;
		
	}

	/**
	 *  Open the specified xls and process it
	 */
	private void execute() {

		try {
			
			
			ddlString ="CREATE OR REPLACE DIRECTORY load_dir AS '"+pwd+"'"+newline+";"+newline+newline; 
			
			POIFSFileSystem fs = new POIFSFileSystem(new FileInputStream(spreadsheet));
			
			HSSFWorkbook wb = new HSSFWorkbook(fs);
			
			processWorkook(wb);
			
			write(ddlString, "ExternalTables.sql");

		
		} catch (Exception e) {
		
			e.printStackTrace();
			
		}

	}

	/**
	 * @param row
	 */
	private List getColumns(HSSFRow names, HSSFRow types) {

		ArrayList cols = new ArrayList();

		for (short c = 0; c < names.getPhysicalNumberOfCells(); c++) {

			HSSFCell namecell = names.getCell(c);
			HSSFCell typecell = types.getCell(c);
			ExternalTableColumn col = new ExternalTableColumn();

			// Now look at the row type row (which should contain
			// an example max value for the column
			col.setName(namecell.getStringCellValue());
			
			if (typecell.getCellType() == HSSFCell.CELL_TYPE_STRING
			 || typecell.getCellType() == HSSFCell.CELL_TYPE_BLANK) 
			{
				col.setLength(typecell.getStringCellValue().length());
				col.setType(HSSFCell.CELL_TYPE_STRING);
				cols.add(col);
			}

			if (typecell.getCellType() == HSSFCell.CELL_TYPE_NUMERIC) 
			{
				
				col.setNumericPrecision(typecell.getNumericCellValue());
				col.setType(HSSFCell.CELL_TYPE_NUMERIC);
				cols.add(col);
			}

		}

		return cols;
	}

	/**
	 * @param sheet
	 * @param table
	 */
	private void processSheet(HSSFSheet sheet, ExternalTable table) {
		
		//Get the table definition information
		table.setColumns(getColumns(sheet.getRow(COLUMN_NAME_ROW), sheet.getRow(COLUMN_TYPE_ROW)));

		//Write out a .csv file based upon the sheet
		writeCsv(sheet, table.getLocation());

		//Add the ddl for the table to the script
		ddlString +=table.getDdl();
	}

	/**
	 * @param wb
	 * Iterate through each sheet in the workbook
	 * and process it
	 */
	private void processWorkook(HSSFWorkbook wb) {

		for (int i = 0; i < wb.getNumberOfSheets(); i++) 
		{
			HSSFSheet sheet = wb.getSheetAt(i);
		
			ExternalTable table = new ExternalTable(wb.getSheetName(i));
			
			processSheet(sheet, table);
			
			System.out.println("...Table "+ table.getName()+ " processed." );
		}
		
	}

	/**
	 * @param content
	 * @param filename
	 * Write the given String content to the file system
	 * using the String filename specified
	 */
	private void write(String content, String filename) {

		try {
			File f = new File(filename);
			f.createNewFile();
			FileWriter fr = new FileWriter(filename);
			fr.write(content);
			fr.flush();
			fr.close();
			
			System.out.println("...File " + filename + " created.");
			
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * @param sheet
	 * @param location
	 */
	private void writeCsv(HSSFSheet sheet, String location) {
		
		String csv = "";
		
		//skip putting the column names and type length row in the csv
		for (int r = COLUMN_NAME_ROW + 2; r < sheet.getPhysicalNumberOfRows(); r++) {
			
			for (short c = 0; c < sheet.getRow(r).getPhysicalNumberOfCells(); c++) {
				
				HSSFCell cell = sheet.getRow(r).getCell(c);
				
				String value = null;

				if (cell == null) continue;

				switch (cell.getCellType()) 
				{

					case HSSFCell.CELL_TYPE_NUMERIC:
						value = "" + cell.getNumericCellValue();
						break;

					case HSSFCell.CELL_TYPE_STRING:
						value = cell.getStringCellValue();
						break;

					default:
				}
				csv += value + ",";

			}
			csv+=newline;
		}
		
		//final newline causes problems so remove it
		write(csv.substring(0,csv.length()-1), location);
	}
}